/** @file   utils.h
 * @brief   Declaration of Utils - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_UTILS_H
#define H_WWW_UTILS_H

#include "vec2d.h"
#include <map>
#include <iostream>
#include "objectid.h"
#include "www_map.h"


namespace WeWantWar {


// Just let the compiler know about these classes
class GameObject;



/** @class  Utils
 * @brief   Has collection of static utility functions.
 * @author  Tomi Lamminsaari
 *
 */
class Utils
{
public:

  ///
  /// Static members
  /// ==============
  
  /** Calculates the angle of the vector from point 'origo' to 'p'.
   * @param     origo             The position of the "origo"
   * @param     p                 The other point.
   * @return    Angle of the vector from 'origo' to 'p'. 0 means directly
   *            up and 256 makes the full circle. The type of this value
   *            Allegro's fixed-integer.
   */
  static fixed findVecAngle( const eng2d::Vec2D& origo, const eng2d::Vec2D& p );

  /** Calculates the turning direction when object is standing at position
   * 'from' and facing to direction 'fromAngle' and he is supposed to turn
   * facing towards the coordinate 'to'.
   * @param     from              Object's current position.
   * @param     to                The target position.
   * @param     fromAngle         The direction the object is currently facing.
   * @return    -1 = turn counter-clockwise, 0 = no need to turn, 1 = turn
   *            clockwise.
   */
  static int findTurningDir( const eng2d::Vec2D& from, const eng2d::Vec2D& to,
                             int fromAngle );
                             
  /** Tells if object is facing towards another object.
   * @param     pObj1             Object 1. The object we're interested in.
   * @param     pObj2             Object 2. The another object.
   * @param     factor            Factor that defines how exactly the 'pObj1'
   *                              must face the 'pObj2' until it's interpretted
   *                              as they are facing.
   * @return    'true' if 'pObj1' is facing the 'pObj2'.
   */
  static bool isFacingObject( const GameObject* pObj1, const GameObject* pObj2,
                              float factor );
                              
  /** Tells if object is facing a coordinate
   * @param     pObj              The object in concern
   * @param     pos               The coordinate
   * @param     factor            Factor that defines how exactly the 'pObj'
   *                              must be facing the coordinate 'pos' until
   *                              it's interpretted as they are facing
   * @return    <code>true</code> if object is facing the coordinate
   */
  static bool isFacingCoordinate( const GameObject* pObj,
                                  const eng2d::Vec2D& pos, float factor );

                              
  /** Calculates the distance between two GameObjects
   * @param     pObj1             Pointer to first object
   * @param     pObj2             Pointer to another object.
   * @return    Distance between the objects in pixels.
   */
  static float objectDistance( const GameObject* pObj1, const GameObject* pObj2 );
  
  
  /** Checks if the two given object can see each other. Objects can see each
   * other if there is no obstackle between them and the distance is less than
   * the given value.
   * @param     pObj1             First object
   * @param     pObj2             Another object
   * @param     maxDist           The maximum distance where the objects still
   *                              can see each other.
   * @return    true, if objects can see each other.
   */
  static bool objectsSeeEachOther( const GameObject* pObj1,
                                   const GameObject* pObj2,
                                   float maxDist );
  
  
  
  /** Tells, if any of the living gameobjects is inside the given circle.
   * Living gameobjects are Player, Alien, Civilian.
   * @param     rP                The center of the circle
   * @param     radius            Radius of the circle
   * @return    <code>true</code> if there are objects.
   */
  static bool creatureInsideCircle( const eng2d::Vec2D& rP, float radius );
  
  
  
  /** Tells if there is a Car inside the given circle.
   * @param     rP                Center of the inspection circle
   * @param     radius            Radius of the circle
   * @return    Pointer to the Car - object that is inside the
   *            circle or null-pointer if there are no cars inside
   *            the circle.
   */
  static GameObject* carInsideCircle( const eng2d::Vec2D& rP, float radius );
  
  
  /** Reads the data from given stream as long as given string is encountered.
   * @param     rIn               The input stream.
   * @param     searchFor         The string we search from the stream.
   * @return    <code>true</code> if the string was not found before EOF.
   */
  static bool searchForString( std::istream& rIn, const std::string& searchFor );
  
  
  /** Calculates the sectornumber where given point is inside.
   * @param     rP                The point
   * @return    The sector number where the point lies.
   */
  inline static int sectorNumber( const eng2d::Vec2D& rP )
  {
    if ( Map::getWidth(Map::IN_BLOCKS) <= 80 && Map::getHeight(Map::IN_BLOCKS) <= 80 ) {
      return 1;
    } else {
      return ( static_cast<int>( rP.y() ) / Map::sectorHeight ) * 8 +
               static_cast<int>( rP.x() ) / Map::sectorWidth;
    }
  }
  
  
  /** Reads the requested number of strings from given stream and stores them
   * to a vector.
   * @param     num               Number of strings we should read.
   * @param     rIn               The input stream where we read the strings
   *                              from.
   * @param     rV                The vector where the strings will be pushed.
   * @return    Nonzero if fails.
   */
  static int readStrings( int num, std::istream& rIn,
                          std::vector< std::string >& rV );


  /** A blender-function to use with allegro's sprite functions. Generates
   * a effect for showing the Predator alien.
   * @param     x                 Source color
   * @param     y                 Destination color
   * @param     n                 Interpolation factor.
   * @return    A new color value.
   */
  static unsigned long predator_blend_b16( unsigned long x, unsigned long y,
                                           unsigned long n );
                                           

  /** Checks if the given point is inside the polygon.
   * @param   aVertices         Vector containing the array of the vertices
   * @param   aPoint            The point
   * @return  true if @c aPoint is inside the polygon defined by @c aVertices.
   */
  static bool pointInsidePolygon(const std::vector<eng2d::Vec2D>& aVertices,
                                 const eng2d::Vec2D& aPoint);
  
  
  ///
  /// Here is a set of utility-structures used in the game.
  /// =====================================================
  
  /** @struct   WeaponSpecsConfig
   * @brief     A structure for the Weapon Specs we load from the configuration
   *            file.
   * @author    Tomi Lamminsaari
   *
   * The <code> Settings </code> - class uses this structure when it loads the
   * weapon configuration from file.
   */
  struct WeaponSpecsConfig {
    /** The weapon id-code. Possible values can be found from
     * <code> Weapon </code> - class. Weapon::W_RIFLE for example.
     */
    int weaponTypeID;
    /** This std::map structure has the parameters read from the configuration
     * file. Name of the parameter is they key-member and value is the
     * data-member. Both are std::string type.
     */
    std::map< std::string, std::string > properties;
  };
  
  
  
  /** @struct   GObjectConfig
   * @brief     A structure for the GameObject - settings we load from the
   *            configuration file.
   * @author    Tomi Lamminsaari
   *
   * As we load the settings from the configuration file, the settings for the
   * GameObjects are stored in structures of this type.
   *
   * This structure is used in <code> Settings </code> - class.
   */
  struct GObjectConfig {
    /** ID-Code of the object whose settings this structure has. */
    ObjectID::Type  objectID;
    /** This std::map structure has the parameters of this GameObject.
     * Name of the parameter is the key-member and value is the data-member.
     */
    std::map< std::string, std::string > properties;
  };
  
  
  
  /** class constructor.
   */
  Utils();
  /** class destructor
   */
  ~Utils();
};


};    // end of namespace


#endif // UTILS_H

/**
 * Version history
 * ===============
 * $Log: utils.h,v $
 * Revision 1.3  2006/08/13 21:05:38  lamminsa
 * sectorNumber() method adapts to current map sizes.
 *
 * Revision 1.2  2006/07/13 16:27:26  lamminsa
 * pointInsidePolygon() method added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:19+02  lamminsa
 * Initial revision
 *
 */
 
